/*
 * $QNXLicenseC:
 * Copyright 2010, QNX Software Systems. All Rights Reserved.
 *
 * You must obtain a written license from and pay applicable
 * license fees to QNX Software Systems before you may reproduce,
 * modify or distribute this software, or any work that includes
 * all or part of this software.   Free development licenses are
 * available for evaluation and non-commercial purposes.  For more
 * information visit http://licensing.qnx.com or email
 * licensing@qnx.com.
 *
 * This file may contain contributions from others.  Please review
 * this entire file for other proprietary rights or license notices,
 * as well as the QNX Development Suite License Guide at
 * http://licensing.qnx.com/license-guide/ for other information.
 * $
 */

#include <sys/types.h>
#include <assert.h>
#include <string.h>


#include "platform.h"
#include "cypress_i2c.h"
#include "bootloader.h"
#include "common.h"
#include "system_information.h"
#include "configuration_test.h"
#include "operating_mode.h"

#ifndef UNITTEST
#include <sys/neutrino.h>
#include <sys/iofunc.h>
#include <sys/dispatch.h>
#include <screen/screen.h>
#include "dcmd_pm.h"
#include "panel.h"
#include "lib_display_binder.h"
#else
#include "ipc_header.h"
#include "slog2_header.h"
#include "screen_header.h"
#include "input_utest_header.h"
#include "interrupt_header.h"
#include "i2c_header.h"
#include "touchscreen_header.h"
#include "timer_header.h"
#include <stdbool.h>
#endif

#include "fidm_touch.h"
#include <hw/cypressctrl.h>

#ifdef UNITTEST
#define timer_create mock_timer_create
#define sched_param ut_sched_param
#endif


#ifndef CYPRESS_H_
#define CYPRESS_H_

#ifdef  __cplusplus
extern "C" {
#endif


#define CYPRESS_TP_THREAD_PRIORITY              21

#define CYPRESS_TP_PULSE_PRIORITY               21
#define CYPRESS_TP_PULSE_CODE                   MTOUCH_PULSE_CODE_MINAVAIL

#define CYPRESS_TP_ERROR_PULSE_PRIORITY         21
#define CYPRESS_TP_ERROR_PULSE_CODE             (CYPRESS_TP_PULSE_CODE+1)

#define CYPRESS_MODE_CHG_TIMEOUT_PULSE_CODE             (CYPRESS_TP_PULSE_CODE+8)

#define CYPRESS_RELEASE_TIMEOUT_EVENT_PULSE_CODE (CYPRESS_TP_PULSE_CODE+10) /* We do +10 to give room for any other pulses that may get used */

#define CYPRESS_I2C_SLAVE_ADDR                  0x4
#define DEFAULT_PM_DEV_NODE                     "/dev/pm"
#define TP_I2C_SPD 400000

#define CYPRESS_TOUCH_INVALID 0xFFF /* The touch value when a finger is not touching */
#define CYPRESS_TOUCH_MASK 0xFFF

#define CYPRESS_I2C_DATA_READ_LENGTH        10
#define CYPRESS_MAX_REGLEN 255

#define CYPRESS_MAX_SAMPLE_READ_SIZE    MAX_I2C_LEN
#define CYPRESS_VALID       0x80
#define CYPRESS_CONTACT_ID  0x7c
#define CYPRESS_CONTACT_ID_SHIFT  2
#define CYPRESS_IN_RANGE    0x02
#define CYPRESS_TOUCH_DOWN  0x01

/* The following are offsets relative to the individual touch slots */
#define CYPRESS_STATUS       1
#define CYPRESS_COORD_X_LOW  2
#define CYPRESS_COORD_X_HIGH 3
#define CYPRESS_COORD_Y_LOW  4
#define CYPRESS_COORD_Y_HIGH 5
#define CYPRESS_COORD_Z_LOW  6
#define CYPRESS_COORD_Z_HIGH 7
#define CYPRESS_COORD_REG_SIZE 1

/* Button Defines */
#define CYPRESS_BUTTONS_PER_REG 4

#define CYPRESS_REPORT_ID_VENDOR_SPECIFIC  0x03
#define CYPRESS_REPORT_ID_PARALLEL_MODE        0x06

/* Driver States */
#define FIRMWARE_UPGRADE          0
#define BOOTLOADER                1
#define SYSTEM_INFORMATION        2
#define OPERATING                 3
#define UNKNOWN                   9

/* Bootloader Firmware */
#define MAX_RETRY_LIMIT      15

/* Name Attach */
#define ATTACH_POINT         "cypress_ctrl_channel"

/* power register values */
#define CYPRESS_POWER_WAKE  0x0
#define CYPRESS_POWER_SLEEP 0x1

#define CYPRESS_MAX_RESET_ATTEMPTS 7

#define CYPRESS_RESET_DELAY_MS   400

/* Configuration and Test mode limits */
#define CYPRESS_CONFIGURATION_MODE_MAX_REPLY_ATTEMPTS  50
#define CYPRESS_CONFIGURATION_MODE_REPLY_DEFAULT_DELAY 2

/* Operating Mode Limits */
#define CYPRESS_OPERATING_MODE_MAX_REPLY_ATTEMPTS      3
#define CYPRESS_OPERATING_MODE_REPLY_DEFAULT_DELAY     50

#define CYPRESS_CORE_WAKEUP_TIMEOUT                     500

/* Watchdog */
#define CYPRESS_HEARTBEAT_PULSE_CODE        _PULSE_CODE_MINAVAIL
#define CYPRESS_BOOT_WATCHDOG_PULSE_CODE    (CYPRESS_TP_PULSE_CODE+4)
/* Pulse codes for receiving pulses from power manager */
#define CYPRESS_PM_PREPARE_PULSE_CODE       (_PULSE_CODE_MINAVAIL+5)
#define CYPRESS_PM_SUSPEND_PULSE_CODE       (_PULSE_CODE_MINAVAIL+6)
#define CYPRESS_PM_RESUME_PULSE_CODE        (_PULSE_CODE_MINAVAIL+7)

#define CYPRESS_DEFAULT_HEARTBEAT_SLEEP     2 /* sec */
#define CYPRESS_MIN_BOOT_WATCHDOG_SLEEP 3

#define CYPRESS_DEFAULT_TOUCHPOINTS 2
#define CYPRESS_MAX_TOUCHPOINTS     15
#define CYPRESS_NUM_TOUCHPOINTS     10

#define CYPRESS_SAMPLE_READ_SIZE(cypress) ((cypress->touch_reg_count * CYPRESS_COORD_REG_SIZE) + 2) /* +2 to include status and counter regs */

#define I2C_TRANSACT_RETRIES 5
#define I2C_TRANSACT_RETRY_DELAY_MS 5

#define I2C_BYTE_ADDRESSING  1
#define I2C_SHORT_ADDRESSING 2

#define MSEC                 1000000

#define CYPRESS_DEFAULT_SLOG_CHAR_LIMIT 80

#define PENDING_COUNT_INFINEON 6
#define REPEAT_COUNT_INFINEON  2
#define CAT_MODE_CHANGE_RE_REQUEST -10

#define DEV_CONTROL_PATH     "dev/vcd/display-binder/control"

typedef struct {
    _Uint8t status;
    _Uint8t bl_id;
    _Uint8t bl_ver;
    _Uint8t app_start[2];
    _Uint8t app_cksum[2];
    _Uint8t app_ver[2];
    _Uint8t silicon_id[8]; /* First 2 bytes are Silicon ID - 0x586 for Cypress TMA301E */
    _Uint8t build_id[4];
} boot_block_t;

typedef enum {
    CYPRESS_POWER_STATE_NONE = 0,
    CYPRESS_POWER_STATE_AWAKE,
    CYPRESS_POWER_STATE_ASLEEP
} cypress_power_state_e;

typedef struct cypress_system_information_memory_map {
    uint8_t hst_mode;
    uint8_t reset_detect;
    uint16_t map_sz;           /* Map size */
    uint16_t cydata_ofs;       /* Cypress Data offset */
    uint16_t test_ofs;         /* Test offset */
    uint16_t pcfg_ofs;         /* Panel Configuration ofset */
    uint16_t opcfg_ofs;        /* Operational Mode info offset */
    uint16_t ddata_ofs;        /* Design Data offset */
    uint16_t mdata_ofs;        /* Manufacturer Data offset */
} cypress_system_information_memory_map_t;

typedef struct cypress_register_map {
    cypress_system_information_memory_map_t sys_info_mem_map;
    cypress_data_t *cypress_data;
    uint8_t *system_information_mode_reg;
    operating_mode_info_t *operating_mode_info;
    uint8_t *operating_mode_reg;
    pcfg_data_t pcfg_data;
    customer_data_t customer_data;
} cypress_register_map_t;

typedef struct cypress_dev {
    struct mtouch_device* inputevents_hdl;

    /* Generic params */
    unsigned touch_reg_count;
    unsigned width;
    unsigned height;
    unsigned invert_y;
    unsigned invert_x;
    unsigned protocol;
    unsigned num_touchpts;
    unsigned verbose;

    /* Driver state */
    unsigned driver_state;
    unsigned delay_driver;
    bool controller_reset_state;

    /* Bootloader state */
    unsigned bootloader_state;

    /* I2C */
    char* i2c_lib;
    unsigned i2c_slave_addr;
    char* i2c_dev_name;
    int i2c_low_clock_tweak; /* ns */
    int i2c_high_clock_tweak; /* ns */
    void* i2c_dev;
    unsigned i2c_speed;

    /* qc I2C */
    int i2c_fd;
    qc_i2c_funcs_t i2c_funcs;

    /* TP interrupt */
    unsigned tp_intr;
    int tp_iid;
    struct sigevent tp_intrevent;
    unsigned tp_err_intr;
    int tp_err_iid;
    struct sigevent tp_err_intrevent;
    pthread_t recv_thread;
    pthread_t fidm_thread;
    unsigned recv_thread_prio;
    unsigned pulse_prio;
    int coid;

    char *attach_point;
    char *fidm_attach_point;

    /* Polling (Debug) */
    unsigned polling_rate;
    timer_t timer_id;

    name_attach_t *attach;
    boot_block_t boot_block;
    _Uint32t seq_id;

    /* Powerman interface */
    cypress_power_state_e power_state;
    /* pm */
    char* pm_dev;
    int pm_fd;
    enum pm_state cy_pm_state;
    enum pm_state cy_str_state;

    /* screen */
    screen_context_t screen_ctx;
    screen_window_t window;

    /* window */
    unsigned int win_height;
    unsigned int win_width;
    unsigned int win_xpos;
    unsigned int win_ypos;
    unsigned int disp_id;
    char* win_name;

     /*display binder details*/
    int display_group_id;
    char* dev_ctrl_path;
    char* dev_status_path;

    /* Debug interface */
    unsigned debug_port;
    unsigned debug_priority;
    pthread_t debug_thread;
    unsigned debug_retry_count;
    unsigned debug_retry_interval;
    int debug_raw_mode; /* We mask the interrupt, prevent resets and don't set the report threshold when in RAW mode */

    /* Platform specific data */
    platform_specific_t platform;
    unsigned reset_gpio_base_addr;
    unsigned reset_gpio_pin;
    uint32_t intr_gpio_pin;
    unsigned err_intr_gpio_pin;

    /* Controller Specific */
    cypress_register_map_t register_map;
    unsigned reset_delay;
    unsigned reset_recovery_delay;
    unsigned soft_reset_recovery_delay;
    unsigned configuration_reply_retry_limit;
    unsigned configuration_reply_retry_delay;
    unsigned operating_mode_reply_retry_limit;
    unsigned operating_mode_reply_retry_delay;
    unsigned command_reply_retry_limit;
    unsigned command_reply_retry_delay;

    /* Firmware update */
    unsigned update_firmware;
    char *firmware_file;
    uint8_t sensing_mode;
    uint8_t chip_revision;
    uint8_t file_chip_revision;
    uint8_t silicon_gen;
    uint16_t silicon_id;
    uint16_t file_silicon_id;
    struct cyacd_record *head_record;
    struct cyacd_record *curr;
    int firmware_update_state;
    int program_record_state;
    int retry_attempts;
    int retry_limit;
    int firmware_rcvid;
    int firmware_line_count;            // Used for progress
    int firmware_lines_complete;
    int calibration_required;
    int panel_detach_detected;
    int idac_calibration_status;

    /* Protocol Specific */
    touch_report_t touch_report[CYPRESS_MAX_TOUCHPOINTS];
    point_t last_coords;
    /*Array for storing processed touch events */
    uint8_t num_prev_recs[CYPRESS_NUM_TOUCHPOINTS];

    /* Controller Heartbeat */
    unsigned heartbeat_sleep;
    struct sigevent heartbeat_event;
    timer_t heartbeat_timerid;
    struct itimerspec itime;
    uint8_t controller_reset_count;

    /* Bootloader watchdog */
    unsigned boot_watchdog_sleep;
    struct sigevent boot_watchdog_event;
    timer_t boot_watchdog_timerid;

    /* Mode change timer */
    unsigned mode_change_sleep;
    struct sigevent mode_change_event;
    timer_t mode_change_timerid;

    /* Release inject check */
    unsigned release_timeout;
    struct sigevent inject_release_event;
    timer_t *inject_release_timerid;
    touch_report_t *last_touch_report;

    /* CapSense Button states */
    uint8_t capsense_button_state[16];
    uint8_t capsense_button_old_state[16];
    int app_rcvid;
    struct sigevent app_notify_event;
    int num_button_regs;

    /* Serializer (if used) */
    unsigned serializer_slave_addr;
    char* serializer_i2c_bus;
    unsigned serializer_i2c_speed;

    /* Logging */
    char *pps_status_obj;
    int pps_status_fd;
    unsigned slog_char_limit;

    /* Manufacturing tests */
    /* Cm Panel Test */
    unsigned run_cm_panel_test;
    unsigned **cm_panel_ref_max;
    unsigned **cm_panel_ref_min;
    unsigned cm_panel_tolerance_max;
    unsigned cm_panel_tolerance_min;
    /* Cp Panel Test */
    unsigned run_cp_panel_test;
    unsigned *cp_panel_tx_ref_max;
    unsigned *cp_panel_tx_ref_min;
    unsigned *cp_panel_rx_ref_max;
    unsigned *cp_panel_rx_ref_min;
    unsigned cp_panel_tolerance_max;
    unsigned cp_panel_tolerance_min;
    /* Save Cm & Cp Panel Test Data */
    unsigned save_cm_cp_data;
    unsigned run_calib_cm_fail;
    unsigned run_calib_cp_fail;

    /* Panel Detach Workaround */
    unsigned detect_panel_detach;
    unsigned panel_detach_cp_value;
    unsigned panel_detach_cp_count;
    unsigned panel_detach_cm_value;
    unsigned panel_detach_cm_count;

    /* Configuration & Test Mode details */
    uint16_t config_block_size;

    /* size details */
    uint8_t opcfg_size;

    /*hard reset */
    unsigned int hard_reset;

    /* Error Pin Status */
    uint16_t error_pin_status;

    /* touch diagnostic details */
    name_attach_t *fidm_attach;
    int fidm_coid;
    uint8_t rtd_enable;
    uint8_t rtd_data[MAX_RTD_DATA_LENGTH];
    int rtd_len;
    bool rtd_readflag;
    int intr_coid;

    pthread_mutex_t tp_event_mutex;
    pthread_mutex_t ctrl_reset_mutex;

    bool haptic_flag;

    int ext_recvid;
    unsigned sync_point;
    char* log_name;
    uint8_t repeatCount;
    bool play_haptic_config;
    unsigned retry_init_attempts;
#ifdef BOSCH_RTC_2487095_HID_CYPRESS_DATASHEET_ALIGNMENT_CHANGES
    bool sysinfo_to_operating_mode_state;
    bool cat_to_operating_mode_state;
    bool haptic_write_status;
#endif
} cypress_dev_t;

typedef struct cypress_touch_state {
    uint8_t status;
    uint8_t id;
    uint16_t x;
    uint16_t y;
    uint16_t reserved1;
    uint16_t reserved2;
} cypress_touch_state_t;

typedef struct cypress_touch_packet {
    uint8_t status;
    uint8_t id;
    uint16_t x;
    uint16_t y;
    uint16_t reserved1;
    uint16_t reserved2;
} cypress_touch_packet_t;

/* Interrupt */
int tp_intr_attach(cypress_dev_t *cypress, struct sigevent *intrevent, int intr, int pulse_code);
int tp_intr_gpio_to_irq(cypress_dev_t *cypress);

/* Common Functions */
int cypress_get_mode(cypress_dev_t* cypress);
int cypress_set_mode(cypress_dev_t* cypress, uint8_t buf);
int cypress_soft_reset(cypress_dev_t* cypress);
int cypress_set_heartbeat_timer(cypress_dev_t* cypress, unsigned sec);
int cypress_display_packet(cypress_dev_t* cypress, char *direction, uint8_t reg, uint8_t *packet, int len);
int cypress_reply_ready(cypress_dev_t* cypress);
int cypress_read_command_reply(cypress_dev_t* cypress, uint8_t **reply, uint16_t len);
void cypress_update_pps_status(cypress_dev_t* cypress, char *status);
int cypress_check_error_pin(cypress_dev_t* cypress);

/* Bootloader Functions */
uint16_t calculate_bootloader_crc(uint8_t *buf, int len);
int cypress_bootloader_cmd(cypress_dev_t* cypress, uint8_t cmd, uint8_t addr, uint8_t *data, uint8_t data_len);
int cypress_bootloader_update_firmware (cypress_dev_t* cypress);
int bootloader_state_machine(cypress_dev_t *cypress);
int cypress_bootloader_open_file(cypress_dev_t* cypress);
int cypress_set_boot_watchdog_timer(cypress_dev_t* cypress, unsigned sec);
int cypress_bootloader_clear_error_pin(cypress_dev_t* cypress);

/* System information Functions */
int cypress_read_cypress_data(cypress_dev_t* cypress);
int cypress_map_operating_mode_registers(cypress_dev_t* cypress);
int cypress_read_memory_map(cypress_dev_t* cypress);
void system_information_clean_up(cypress_dev_t* cypress);
void cypress_dump_pcfg_data(cypress_dev_t* cypress);
void cypress_dump_opmode_data(cypress_dev_t* cypress);
void cypress_dump_cypress_data(cypress_dev_t* cypress);
void cypress_dump_memory_map_data(cypress_dev_t* cypress);

/* Operationg Mode Functions */
int cypress_operating_mode_cmd(cypress_dev_t* cypress, uint8_t cmd, uint8_t *data, uint8_t data_len, uint8_t **reply, uint8_t reply_len);
int cypress_operating_mode_heartbeat(cypress_dev_t* cypress);
int cypress_operating_mode_fetch_noise_levels(cypress_dev_t* cypress, uint16_t *noise_level, uint16_t *noise_level_velocity);
int cypress_play_haptic_data(cypress_dev_t* cypress, uint8_t *haptic_effect);

/* Reset function */
int cypress_controller_reset(cypress_dev_t* cypress);

/* Configuration Test Mode Functions */
int cypress_calibrate_idacs(cypress_dev_t* cypress);
int cypress_config_cmd(cypress_dev_t* cypress, uint8_t cmd, uint8_t *data, uint8_t data_len, uint8_t **reply, uint16_t reply_len);
int16_t cypress_retrieve_idac_read_length(cypress_dev_t* cypress);
int cypress_retrieve_idac_data(cypress_dev_t* cypress, uint8_t **reply);
int cypress_init_baselines(cypress_dev_t* cypress);
int cypress_run_cm_panel_test(cypress_dev_t* cypress, int skip_setting_mode, int save_to_file);
int cypress_run_cp_panel_test(cypress_dev_t* cypress, int skip_setting_mode, int save_to_file);
int cypress_read_config_param_flash(cypress_dev_t* cypress, uint8_t addr, uint8_t *data, uint8_t len);
int cypress_config_clear_error_pin(cypress_dev_t* cypress);
int cypress_set_haptic_data(cypress_dev_t* cypress, uint8_t *haptic_data, uint16_t len);
int cypress_initialize_haptic_cmd(cypress_dev_t* cypress, uint8_t *buff, uint16_t len);

int cypress_wake(cypress_dev_t* cypress);
int cypress_sleep(cypress_dev_t* cypress);

/* register to pm function */
int cypress_register_pm(cypress_dev_t* cypress);
void* cypress_ext_msg_handler(void *arg);

int cypress_process_button(cypress_dev_t* cypress);
int cypress_process_touch(cypress_dev_t* cypress, int no_touch_records);
int cypress_create_qvm_window(cypress_dev_t *dev);
void cypress_release_touch(cypress_dev_t *dev);
void error_memory(const char * fmt, ...);
int cypress_get_display_grpid(cypress_dev_t* cypress);
int cypress_driver_haptic_mode_check(cypress_dev_t* cypress);
#ifdef BOSCH_RTC_2487095_HID_CYPRESS_DATASHEET_ALIGNMENT_CHANGES
bool cypress_process_mode_change_interrupt(cypress_dev_t* cypress);
#endif

#ifdef  __cplusplus
}
#endif

#endif /* CYPRESS_H_ */

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn.ott.qnx.com/product/branches/7.0.0/trunk/hardware/mtouch/cypress/touch.h $ $Rev: 888004 $")
#endif
